---
name: v.db.update.py
description: Updates a column in the attribute table connected to a vector map.
keywords: [ vector, attribute table, database, attribute update, type casting ]
---

# v.db.update.py

Updates a column in the attribute table connected to a vector map.

=== "Command line"

    **v.db.update.py**
    **map**=*name*
    **layer**=*string*
    **column**=*name*
    [**value**=*string*]
    [**query_column**=*name*]
    [**where**=*sql_query*]
    [**sqliteextra**=*name*]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    v.db.update.py map=name layer=1 column=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***v.db.update.py***",
        **map**,
        **layer**=*"1"*,
        **column**,
        **value**=*None*,
        **query_column**=*None*,
        **where**=*None*,
        **sqliteextra**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("v.db.update.py", map="name", layer="1", column="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.v_db_update_py*(**map**,
        **layer**=*"1"*,
        **column**,
        **value**=*None*,
        **query_column**=*None*,
        **where**=*None*,
        **sqliteextra**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.v_db_update_py(map="name", layer="1", column="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **map**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    **layer**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **column**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of attribute column to update  
    **value**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Literal value to update the column with  
    **query_column**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of other attribute column to query, can be combination of columns (e.g. co1+col2)  
    **where**=*sql_query*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL statement without 'where' keyword  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: income &lt; 1000 and population &gt;= 10000  
    **sqliteextra**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of SQLite extension file for extra functions (SQLite backend only)  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **map** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **layer** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **column** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of attribute column to update  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **value** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Literal value to update the column with  
    **query_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of other attribute column to query, can be combination of columns (e.g. co1+col2)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **where** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL statement without 'where' keyword  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: income &lt; 1000 and population &gt;= 10000  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, sql_query, *sql_query*  
    **sqliteextra** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of SQLite extension file for extra functions (SQLite backend only)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **map** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **layer** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **column** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of attribute column to update  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **value** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Literal value to update the column with  
    **query_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of other attribute column to query, can be combination of columns (e.g. co1+col2)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **where** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL statement without 'where' keyword  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: income &lt; 1000 and population &gt;= 10000  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, sql_query, *sql_query*  
    **sqliteextra** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of SQLite extension file for extra functions (SQLite backend only)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*v.db.update* assigns a new value to a column in the attribute table
connected to a given map. The *value* parameter allows updating with a
literal value. Alternatively, with the *qcol* parameter values can be
copied from another column in the table or be the result of a
combination or transformation of other columns.

## NOTES

*v.db.update* is just a front-end to *db.execute* to allow easier usage.

For complex SQL UPDATE statements, *db.execute* should be used.

## EXAMPLES

### Replacing of NULL values

In this example, selectively display lakes without (blue) and with NULL
(red) are shown to find out which type is undefined. In the original map
there are lakes missing FTYPE attribute which are wetlands along
streams. These NULL attributes are replaced with the landuse type
WETLAND:

```sh
g.copy vect=lakes,mylakes
v.db.select mylakes
v.db.select mylakes where="FTYPE IS NULL"

# display the lakes, show undefined FTYPE lakes in red
g.region vector=mylakes
d.mon wx0
d.vect mylakes where="FTYPE NOT NULL" type=area col=blue
d.vect mylakes where="FTYPE IS NULL" type=area col=red

# replace NULL with FTYPE WETLAND
v.db.update mylakes col=FTYPE value=WETLAND \
            where="FTYPE IS NULL"
v.db.select mylakes
```

### Updating of columns with on the fly calculation

Spearfish example: adding new column, copying values from another table
column with on the fly calculation:

```sh
g.copy vect=fields,myfields
v.db.addcolumn myfields col="polynum integer"
v.db.update myfields col=polynum qcol="cat*2"
v.db.select myfields
```

### Type casting

Type cast (type conversion) of strings to double precision (unsupported
by DBF driver):

```sh
g.copy vect=geodetic_pts,mygeodetic_pts
v.db.update mygeodetic_pts col=zval qcol="CAST(z_value AS double precision)" \
            where="z_value <> 'N/A'"
```

### Updating of columns with on the fly calculation (SQLite extended functions)

Note: this requires SQLite extended functions. For details see the GRASS
Wiki (compilation of
[libsqlitefunctions.so](https://grasswiki.osgeo.org/wiki/Build_SQLite_extension_on_Linux)
and
[libsqlitefunctions.dll](https://grasswiki.osgeo.org/wiki/Build_SQLite_extension_on_windows)).

North Carolina data set example: adding new column, copying values from
another table column with on the fly calculation:

```sh
g.copy vect=precip_30ynormals,myprecip_30ynormals
v.db.addcolumn myprecip_30ynormals column="logjuly double precision"
v.db.update myprecip_30ynormals column="logjuly" query_column="log(jul)" \
  sqliteextra=$HOME/sqlite_extensions/libsqlitefunctions.so

v.db.select myprecip_30ynormals columns=jul,logjuly
jul|logjuly
132.842|4.88916045210132
127|4.84418708645859
124.206|4.82194147751127
104.648|4.65060233738593
98.298|4.58800368106618
...
```

## SEE ALSO

*[db.execute](db.execute.md), [v.db.addcolumn](v.db.addcolumn.md),
[v.db.addtable](v.db.addtable.md), [v.db.connect](v.db.connect.md),
[v.db.droptable](v.db.droptable.md), [v.db.join](v.db.join.md),
[v.db.select](v.db.select.md)*

*[GRASS SQL interface](sql.md)*

## AUTHOR

Moritz Lennert (<mlennert@club.worldonline.be>)

## SOURCE CODE

Available at: [v.db.update source code](https://github.com/OSGeo/grass/tree/main/scripts/v.db.update)
([history](https://github.com/OSGeo/grass/commits/main/scripts/v.db.update))  
Latest change: Friday Aug 08 09:39:19 2025 in commit [ed72c71](https://github.com/OSGeo/grass/commit/ed72c71d8580e872cd6fd345c84a15c324764465)
