---
name: d.legend
description: Displays a legend for a 2D or 3D raster map in the active frame of the graphics monitor.
keywords: [ display, cartography, legend ]
---

# d.legend

Displays a legend for a 2D or 3D raster map in the active frame of the graphics monitor.

=== "Command line"

    **d.legend**
    [**-vctnsfdbl**]
    [**raster**=*name*]
    [**raster_3d**=*name*]
    [**title**=*string*]
    [**title_fontsize**=*float*]
    [**lines**=*integer*]
    [**thin**=*integer*]
    [**units**=*string*]
    [**labelnum**=*integer*]
    [**label_values**=*float* [,*float*,...]]
    [**label_step**=*float*]
    [**digits**=*integer*]
    [**at**=*bottom,top,left,right*]
    [**use**=*float* [,*float*,...]]
    [**range**=*min,max*]
    [**color**=*name*]
    [**font**=*string*]
    [**fontsize**=*float*]
    [**path**=*name*]
    [**charset**=*string*]
    [**border_color**=*name*]
    [**bgcolor**=*name*]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    d.legend raster=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***d.legend***",
        **raster**=*None*,
        **raster_3d**=*None*,
        **title**=*None*,
        **title_fontsize**=*None*,
        **lines**=*0*,
        **thin**=*1*,
        **units**=*None*,
        **labelnum**=*5*,
        **label_values**=*None*,
        **label_step**=*None*,
        **digits**=*None*,
        **at**=*None*,
        **use**=*None*,
        **range**=*None*,
        **color**=*"black"*,
        **font**=*None*,
        **fontsize**=*None*,
        **path**=*None*,
        **charset**=*None*,
        **border_color**=*"black"*,
        **bgcolor**=*"white"*,
        **flags**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("d.legend", raster="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.d_legend*(**raster**=*None*,
        **raster_3d**=*None*,
        **title**=*None*,
        **title_fontsize**=*None*,
        **lines**=*0*,
        **thin**=*1*,
        **units**=*None*,
        **labelnum**=*5*,
        **label_values**=*None*,
        **label_step**=*None*,
        **digits**=*None*,
        **at**=*None*,
        **use**=*None*,
        **range**=*None*,
        **color**=*"black"*,
        **font**=*None*,
        **fontsize**=*None*,
        **path**=*None*,
        **charset**=*None*,
        **border_color**=*"black"*,
        **bgcolor**=*"white"*,
        **flags**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.d_legend(raster="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **raster**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of raster map  
    **raster_3d**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of 3D raster map  
    **title**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Legend title  
    **title_fontsize**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Title font size  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: Same as fontsize  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-360*  
    **lines**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of text lines (useful for truncating long legends)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0-1000*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **thin**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Thinning factor (thin=10 gives cats 0,10,20...)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-1000*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **units**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Units to display after labels (e.g. meters)  
    **labelnum**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of text labels for smooth gradient legend  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *2-100*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *5*  
    **label_values**=*float* [,*float*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Specific values to draw ticks  
    **label_step**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Display label every step  
    **digits**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of digits after decimal point  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0-6*  
    **at**=*bottom,top,left,right*  
    &nbsp;&nbsp;&nbsp;&nbsp;Size and placement as percentage of screen coordinates (0,0 is lower left)  
    &nbsp;&nbsp;&nbsp;&nbsp;bottom,top,left,right  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0-100*  
    **use**=*float* [,*float*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;List of discrete category numbers/values for legend  
    **range**=*min,max*  
    &nbsp;&nbsp;&nbsp;&nbsp;Use a subset of the map range for the legend (min,max)  
    **color**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Text color  
    &nbsp;&nbsp;&nbsp;&nbsp;Either a standard color name or R:G:B triplet  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *black*  
    **font**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Font name  
    **fontsize**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Font size  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: Auto-scaled  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-360*  
    **path**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Path to font file  
    **charset**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Text encoding (only applicable to TrueType fonts)  
    **border_color**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Border color  
    &nbsp;&nbsp;&nbsp;&nbsp;Either a standard color name, R:G:B triplet, or "none"  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *black*  
    **bgcolor**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Background color  
    &nbsp;&nbsp;&nbsp;&nbsp;Either a standard color name, R:G:B triplet, or "none"  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *white*  
    **-v**  
    &nbsp;&nbsp;&nbsp;&nbsp;Do not show category labels  
    **-c**  
    &nbsp;&nbsp;&nbsp;&nbsp;Do not show category numbers  
    **-t**  
    &nbsp;&nbsp;&nbsp;&nbsp;Draw legend ticks for labels  
    **-n**  
    &nbsp;&nbsp;&nbsp;&nbsp;Skip categories with no label  
    **-s**  
    &nbsp;&nbsp;&nbsp;&nbsp;Draw smooth gradient  
    **-f**  
    &nbsp;&nbsp;&nbsp;&nbsp;Flip legend  
    **-d**  
    &nbsp;&nbsp;&nbsp;&nbsp;Add histogram to smoothed legend  
    **-b**  
    &nbsp;&nbsp;&nbsp;&nbsp;Show background  
    **-l**  
    &nbsp;&nbsp;&nbsp;&nbsp;Use logarithmic scale  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **raster** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **raster_3d** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of 3D raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster_3d, *name*  
    **title** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Legend title  
    **title_fontsize** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Title font size  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: Same as fontsize  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-360*  
    **lines** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of text lines (useful for truncating long legends)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0-1000*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **thin** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Thinning factor (thin=10 gives cats 0,10,20...)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-1000*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **units** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Units to display after labels (e.g. meters)  
    **labelnum** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of text labels for smooth gradient legend  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *2-100*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *5*  
    **label_values** : float | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Specific values to draw ticks  
    **label_step** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Display label every step  
    **digits** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of digits after decimal point  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0-6*  
    **at** : tuple[float, float, float, float] | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Size and placement as percentage of screen coordinates (0,0 is lower left)  
    &nbsp;&nbsp;&nbsp;&nbsp;bottom,top,left,right  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *bottom,top,left,right*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0-100*  
    **use** : float | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;List of discrete category numbers/values for legend  
    **range** : tuple[float, float] | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Use a subset of the map range for the legend (min,max)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *min,max*  
    **color** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Text color  
    &nbsp;&nbsp;&nbsp;&nbsp;Either a standard color name or R:G:B triplet  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, color, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *black*  
    **font** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Font name  
    **fontsize** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Font size  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: Auto-scaled  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-360*  
    **path** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Path to font file  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: file, *name*  
    **charset** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Text encoding (only applicable to TrueType fonts)  
    **border_color** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Border color  
    &nbsp;&nbsp;&nbsp;&nbsp;Either a standard color name, R:G:B triplet, or "none"  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, color, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *black*  
    **bgcolor** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Background color  
    &nbsp;&nbsp;&nbsp;&nbsp;Either a standard color name, R:G:B triplet, or "none"  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, color, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *white*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *v*, *c*, *t*, *n*, *s*, *f*, *d*, *b*, *l*  
    &nbsp;&nbsp;&nbsp;&nbsp;**v**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Do not show category labels  
    &nbsp;&nbsp;&nbsp;&nbsp;**c**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Do not show category numbers  
    &nbsp;&nbsp;&nbsp;&nbsp;**t**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Draw legend ticks for labels  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Skip categories with no label  
    &nbsp;&nbsp;&nbsp;&nbsp;**s**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Draw smooth gradient  
    &nbsp;&nbsp;&nbsp;&nbsp;**f**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Flip legend  
    &nbsp;&nbsp;&nbsp;&nbsp;**d**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Add histogram to smoothed legend  
    &nbsp;&nbsp;&nbsp;&nbsp;**b**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Show background  
    &nbsp;&nbsp;&nbsp;&nbsp;**l**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use logarithmic scale  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **raster** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **raster_3d** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of 3D raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster_3d, *name*  
    **title** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Legend title  
    **title_fontsize** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Title font size  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: Same as fontsize  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-360*  
    **lines** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of text lines (useful for truncating long legends)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0-1000*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **thin** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Thinning factor (thin=10 gives cats 0,10,20...)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-1000*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **units** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Units to display after labels (e.g. meters)  
    **labelnum** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of text labels for smooth gradient legend  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *2-100*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *5*  
    **label_values** : float | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Specific values to draw ticks  
    **label_step** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Display label every step  
    **digits** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of digits after decimal point  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0-6*  
    **at** : tuple[float, float, float, float] | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Size and placement as percentage of screen coordinates (0,0 is lower left)  
    &nbsp;&nbsp;&nbsp;&nbsp;bottom,top,left,right  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *bottom,top,left,right*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0-100*  
    **use** : float | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;List of discrete category numbers/values for legend  
    **range** : tuple[float, float] | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Use a subset of the map range for the legend (min,max)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *min,max*  
    **color** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Text color  
    &nbsp;&nbsp;&nbsp;&nbsp;Either a standard color name or R:G:B triplet  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, color, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *black*  
    **font** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Font name  
    **fontsize** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Font size  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: Auto-scaled  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-360*  
    **path** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Path to font file  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: file, *name*  
    **charset** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Text encoding (only applicable to TrueType fonts)  
    **border_color** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Border color  
    &nbsp;&nbsp;&nbsp;&nbsp;Either a standard color name, R:G:B triplet, or "none"  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, color, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *black*  
    **bgcolor** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Background color  
    &nbsp;&nbsp;&nbsp;&nbsp;Either a standard color name, R:G:B triplet, or "none"  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, color, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *white*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *v*, *c*, *t*, *n*, *s*, *f*, *d*, *b*, *l*  
    &nbsp;&nbsp;&nbsp;&nbsp;**v**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Do not show category labels  
    &nbsp;&nbsp;&nbsp;&nbsp;**c**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Do not show category numbers  
    &nbsp;&nbsp;&nbsp;&nbsp;**t**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Draw legend ticks for labels  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Skip categories with no label  
    &nbsp;&nbsp;&nbsp;&nbsp;**s**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Draw smooth gradient  
    &nbsp;&nbsp;&nbsp;&nbsp;**f**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Flip legend  
    &nbsp;&nbsp;&nbsp;&nbsp;**d**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Add histogram to smoothed legend  
    &nbsp;&nbsp;&nbsp;&nbsp;**b**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Show background  
    &nbsp;&nbsp;&nbsp;&nbsp;**l**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use logarithmic scale  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*d.legend* displays a legend for a user-specified raster map or 3D
raster map layer in the active frame on the graphics monitor.

The legend's default size is based on the dimensions of the active
frame, specifically its height. *d.legend* will only obscure those
portions of the active frame that directly underlie the legend.

## NOTES

When using the **at** to size & place the legend, a user may create a
horizontal legend by making the box wider than it is tall.

Raster maps based on floating point values will display smoothed, from
greatest to smallest value, while categorical raster maps will display
in order, from top to bottom. Horizontal legends will always be
smoothed. If the box is defined with inverted y-values or an inverted
**range**, the legend will automatically flip. If this is not the
desired result, the **-f** flag may be used to flip it back.

If the user attempts to display a very long legend in a relatively short
display frame, the legend may appear in unreadably small text, or even
revert to a smooth gradient legend. Use the **lines**, **thin**,
**use**, **range**, and/or **-n** options to reduce the number of
categories to be displayed, or the **-s** flag to force a smooth
gradient legend.

The **lines** option will display the first number of categories, as
defined by *value*, contained in the raster map. When used with the
**-n** flag, it takes on a new meaning: "up to category \#". When used
with both **thin** and the **-n** flag, its meaning becomes more
obscure. When using **lines**, auto-scaled text similar to "4 of 16
categories" will be placed at the bottom of the legend.

The **thin** option sets the thinning factor. For raster maps with a 0th
category, **thin**=*10* gives cats \[0,10,20,...\]. For raster maps
starting at category 1, **thin**=*10* gives cats \[1,11,21,...\].

The **use** option lets the user create a legend made up of arbitrary
category values. e.g. **use**=*1000,100,10,0,-10,-100,-1000*

The **range** option lets the user define the minimum and maximum
categories to be used in the legend. It may also be used to define the
limits of a smooth gradient legend created from a raster containing
floating point values. Note the color scale will remain faithful to the
category values as defined with *[r.colors](r.colors.md)*, and the
**range** may be extended to the limits defined by the
*[r.colors](r.colors.md)* color map.

The flag **-n** is useful for categorial maps, as it suppresses the
drawing of non-existing categories (otherwise the full range is shown).

Vertical legends produced with *d.legend* will place text labels to the
right of the legend box, horizontal legends will place text below. This
text will be auto-scaled to fit within the frame, reducing the size of
the legend if necessary. Legends positioned with the **at** option will
not auto-scale text, in order to provide more control to the user.
Smaller text may be obtained in this case by reducing the height of the
box or by using the **fontsize** option. The **-c** and **-v** flags may
be used to suppress the display of category numbers and labels
respectively, or used together to suppress all text of categorial raster
maps.

The text produced from floating-point raster maps will automatically
create output with a meaningful number of significant digits. For very
small values, numbers will be expressed in scientific notation,
e.g. "1.7e-9". Option **digits** can be used to determine how many
digits after decimal point will be displayed.

When the **-d** flag is used to display a histogram distribution along
side the smoothed gradient legend, note that the statistics are
calculated on the *current computational region* settings set by
*g.region*. The default **range** however covers the entire natural
bounds of the input map. If the histogram appears empty, check your
region settings.

If the raster map's *units* metadata has been set with the *r.support*
module then it will be displayed along side the legend.

The option **title** will display the custom title at the top of the
legend. In case of vertical legend the title is aligned to the left edge
of legend, in case of horizontal legend the title is aligned to the
center. **title_fontsize** can be used to set the size of legend title.
By default the legend title font size is the same as labels font size.

There are different options to customize displayed labels. The
**labelnum** set the number of labels which are displayed in regular
intervals. The **label_values** will specify the values where the labels
will be displayed. The **label_step** will display labels at values
which are divisible by this value.

The flag **-t** will show ticks at labels.

The flag **-b** will show the background. Options **bgcolor** and
**border_color** can be used to choose the color of border and
background fill.

The flag **-l** will switch to logarithmic scale. In case this flag is
used, the provided step in **label_step** is interpreted in the
logarithmic space.

## EXAMPLE

Displaying the legend along with a histogram (North Carolina Sample
dataset):

```sh
g.region raster=elevation -p
d.rast elevation
d.legend -d elevation
```

![Elevation map with legend](d_legend.png)

Displaying the legend with custom labels and background:

```sh
g.region raster=elevation -p
d.rast elevation
d.legend raster=elevation -t label_step=20 label_values=108 title=Legend -b bgcolor=255:255:204 border_color=gray
```

![Elevation map with custom legend](d_legend_custom_labels_and_background.png)

Displaying the legend with logarithmic scale:

```sh
g.region raster=elevation -p
r.watershed -a elevation=elevation threshold=1000 accumulation=flowacc
d.rast flowacc
d.legend raster=flowacc -t -l label_step=1
```

![Flow accumulation map with logarithmic legend](d_legend_logarithmic.png)

## SEE ALSO

*[d.barscale](d.barscale.md), [d.colortable](d.colortable.md),
[d.font](d.font.md), [d.grid](d.grid.md), [d.rast](d.rast.md),
[d.rast.leg](d.rast.leg.md), [d.text](d.text.md),
[d.vect.thematic](d.vect.thematic.md), [r.reclass](r.reclass.md),
[r.stats](r.stats.md), [r3.stats](r3.stats.md)*

## AUTHORS

Bill Brown, U.S. Army Construction Engineering Research Laboratories  
Late 2002: Rewrite of much of the code. Hamish Bowman, Otago University,
New Zealand  
Additional improvements from various authors

## SOURCE CODE

Available at: [d.legend source code](https://github.com/OSGeo/grass/tree/main/display/d.legend)
([history](https://github.com/OSGeo/grass/commits/main/display/d.legend))  
Latest change: Tuesday Jun 24 05:57:51 2025 in commit [801e552](https://github.com/OSGeo/grass/commit/801e552cdaf8ef40f8760dbbcc5a4a5b83b14d40)
