---
name: r.grow.distance
description: Generates a raster map containing distances to nearest raster features and/or the value of the nearest non-null cell.
keywords: [ raster, distance, proximity ]
---

# r.grow.distance

Generates a raster map containing distances to nearest raster features and/or the value of the nearest non-null cell.

=== "Command line"

    **r.grow.distance**
    [**-mn**]
    **input**=*name*
    [**distance**=*name*]
    [**value**=*name*]
    [**metric**=*string*]
    [**minimum_distance**=*float*]
    [**maximum_distance**=*float*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.grow.distance input=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.grow.distance***",
        **input**,
        **distance**=*None*,
        **value**=*None*,
        **metric**=*"euclidean"*,
        **minimum_distance**=*None*,
        **maximum_distance**=*None*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.grow.distance", input="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_grow_distance*(**input**,
        **distance**=*None*,
        **value**=*None*,
        **metric**=*"euclidean"*,
        **minimum_distance**=*None*,
        **maximum_distance**=*None*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_grow_distance(input="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    **distance**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for distance output raster map  
    **value**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for value output raster map  
    **metric**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Metric  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *euclidean, squared, maximum, manhattan, geodesic*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *euclidean*  
    **minimum_distance**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum distance threshold  
    **maximum_distance**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum distance threshold  
    **-m**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output distances in meters instead of map units  
    **-n**  
    &nbsp;&nbsp;&nbsp;&nbsp;Calculate distance to nearest NULL cell  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **distance** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for distance output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **value** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for value output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **metric** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Metric  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *euclidean, squared, maximum, manhattan, geodesic*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *euclidean*  
    **minimum_distance** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum distance threshold  
    **maximum_distance** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum distance threshold  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *m*, *n*  
    &nbsp;&nbsp;&nbsp;&nbsp;**m**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Output distances in meters instead of map units  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Calculate distance to nearest NULL cell  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **distance** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for distance output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **value** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for value output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **metric** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Metric  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *euclidean, squared, maximum, manhattan, geodesic*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *euclidean*  
    **minimum_distance** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum distance threshold  
    **maximum_distance** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum distance threshold  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *m*, *n*  
    &nbsp;&nbsp;&nbsp;&nbsp;**m**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Output distances in meters instead of map units  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Calculate distance to nearest NULL cell  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.grow.distance* generates raster maps representing the distance to the
nearest non-null cell in the input map and/or the value of the nearest
non-null cell.

## NOTES

The flag **-n** calculates the respective pixel distances to the nearest
NULL cell.

The user has the option of specifying five different metrics which
control the geometry in which grown cells are created, (controlled by
the **metric** parameter): *Euclidean*, *Squared*, *Manhattan*,
*Maximum*, and *Geodesic*.

The *Euclidean distance* or *Euclidean metric* is the "ordinary"
distance between two points that one would measure with a ruler, which
can be proven by repeated application of the Pythagorean theorem. The
formula is given by:

```sh
d(dx,dy) = sqrt(dx^2 + dy^2)
```

Cells grown using this metric would form isolines of distance that are
circular from a given point, with the distance given by the **radius**.

The *Squared* metric is the *Euclidean* distance squared, i.e. it simply
omits the square-root calculation. This may be faster, and is sufficient
if only relative values are required.

The *Manhattan metric*, or *Taxicab geometry*, is a form of geometry in
which the usual metric of Euclidean geometry is replaced by a new metric
in which the distance between two points is the sum of the (absolute)
differences of their coordinates. The name alludes to the grid layout of
most streets on the island of Manhattan, which causes the shortest path
a car could take between two points in the city to have length equal to
the points' distance in taxicab geometry. The formula is given by:

```sh
d(dx,dy) = abs(dx) + abs(dy)
```

where cells grown using this metric would form isolines of distance that
are rhombus-shaped from a given point.

The *Maximum metric* is given by the formula

```sh
d(dx,dy) = max(abs(dx),abs(dy))
```

where the isolines of distance from a point are squares.

The *Geodesic metric* is calculated as geodesic distance, to be used
only in latitude-longitude coordinate reference system. It is
recommended to use it along with the *-m* flag in order to output
distances in meters instead of map units.

If **minimum_distance** is given, all cells with a distance smaller than
**minimum_distance** will be set to NULL.

If **maximum_distance** is given, all cells with a distance larger than
**maximum_distance** will be set to NULL. The resultant output is
equivalent to a buffer.

If both **minimum_distance** and **maximum_distance** are given, the
result will be similar to a doughnut, a restricted belt for a given
distance range. All cells outside this distance range will be set to
NULL.

## EXAMPLES

### Distance from the streams network

North Carolina sample dataset:

```sh
g.region raster=streams_derived -p
r.grow.distance input=streams_derived distance=dist_from_streams
r.colors map=dist_from_streams color=rainbow
```

![Euclidean distance from the streams network in meters (map subset)](r_grow_distance.png)  
*Euclidean distance from the streams network in meters (map subset)*

![r_grow_distance_zoom](r_grow_distance_zoom.png)  
*Euclidean distance from the streams network in meters (detail, numbers
shown with d.rast.num)*

### Distance from sea in meters in latitude-longitude CRS

```sh
g.region raster=sea -p
r.grow.distance -m input=sea distance=dist_from_sea_geodetic metric=geodesic
r.colors map=dist_from_sea_geodetic color=rainbow
```

![Geodesic distances to sea in meters](r_grow_distance_sea.png)  
*Geodesic distances to sea in meters*

## SEE ALSO

*[r.grow](r.grow.md), [r.distance](r.distance.md),
[r.buffer](r.buffer.md), [r.cost](r.cost.md), [r.patch](r.patch.md)*

*[Wikipedia Entry: Euclidean
Metric](https://en.wikipedia.org/wiki/Euclidean_metric)  
[Wikipedia Entry: Manhattan
Metric](https://en.wikipedia.org/wiki/Manhattan_metric)*

## AUTHOR

Glynn Clements

## SOURCE CODE

Available at: [r.grow.distance source code](https://github.com/OSGeo/grass/tree/main/raster/r.grow.distance)
([history](https://github.com/OSGeo/grass/commits/main/raster/r.grow.distance))  
Latest change: Tuesday Feb 18 17:20:26 2025 in commit [688e625](https://github.com/OSGeo/grass/commit/688e625e7db3be9a465ce99ba25f36d2ebdd5aeb)
