---
name: r.distance
description: Locates the closest points between objects in two raster maps.
keywords: [ raster, distance ]
---

# r.distance

Locates the closest points between objects in two raster maps.

=== "Command line"

    **r.distance**
    [**-lon**]
    **map**=*name1,name2* [,*name1,name2*,...]
    [**separator**=*character*]
    [**sort**=*string*]
    **format**=*name*
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.distance map=name1,name2 format=plain
    ```

=== "Python (grass.script)"

    *grass.script.parse_command*("***r.distance***",
        **map**,
        **separator**=*None*,
        **sort**=*None*,
        **format**=*"plain"*,
        **flags**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.parse_command("r.distance", map="name1,name2", format="json")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_distance*(**map**,
        **separator**=*None*,
        **sort**=*None*,
        **format**=*"plain"*,
        **flags**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_distance(map="name1,name2", format="json")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **map**=*name1,name2* [,*name1,name2*,...] **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of two input raster maps for computing inter-class distances  
    **separator**=*character*  
    &nbsp;&nbsp;&nbsp;&nbsp;Field separator  
    &nbsp;&nbsp;&nbsp;&nbsp;Special characters: pipe, comma, space, tab, newline  
    **sort**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Sort output by distance  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: sorted by categories  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *asc, desc*  
    &nbsp;&nbsp;&nbsp;&nbsp;**asc**: Sort by distance in ascending order  
    &nbsp;&nbsp;&nbsp;&nbsp;**desc**: Sort by distance in descending order  
    **format**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, csv, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Human readable text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**csv**: CSV (Comma Separated Values)  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    **-l**  
    &nbsp;&nbsp;&nbsp;&nbsp;Include category labels in the output  
    **-o**  
    &nbsp;&nbsp;&nbsp;&nbsp;Report zero distance if rasters are overlapping  
    **-n**  
    &nbsp;&nbsp;&nbsp;&nbsp;Report null objects as \*  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **map** : list[tuple[str, str]] | tuple[str, str] | list[str] | str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of two input raster maps for computing inter-class distances  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name1,name2*  
    **separator** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Field separator  
    &nbsp;&nbsp;&nbsp;&nbsp;Special characters: pipe, comma, space, tab, newline  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, separator, *character*  
    **sort** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Sort output by distance  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: sorted by categories  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *asc, desc*  
    &nbsp;&nbsp;&nbsp;&nbsp;**asc**: Sort by distance in ascending order  
    &nbsp;&nbsp;&nbsp;&nbsp;**desc**: Sort by distance in descending order  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, csv, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Human readable text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**csv**: CSV (Comma Separated Values)  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *l*, *o*, *n*  
    &nbsp;&nbsp;&nbsp;&nbsp;**l**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Include category labels in the output  
    &nbsp;&nbsp;&nbsp;&nbsp;**o**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Report zero distance if rasters are overlapping  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Report null objects as \*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **map** : list[tuple[str, str]] | tuple[str, str] | list[str] | str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of two input raster maps for computing inter-class distances  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name1,name2*  
    **separator** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Field separator  
    &nbsp;&nbsp;&nbsp;&nbsp;Special characters: pipe, comma, space, tab, newline  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, separator, *character*  
    **sort** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Sort output by distance  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: sorted by categories  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *asc, desc*  
    &nbsp;&nbsp;&nbsp;&nbsp;**asc**: Sort by distance in ascending order  
    &nbsp;&nbsp;&nbsp;&nbsp;**desc**: Sort by distance in descending order  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, csv, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Human readable text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**csv**: CSV (Comma Separated Values)  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *l*, *o*, *n*  
    &nbsp;&nbsp;&nbsp;&nbsp;**l**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Include category labels in the output  
    &nbsp;&nbsp;&nbsp;&nbsp;**o**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Report zero distance if rasters are overlapping  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Report null objects as \*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.distance* locates the closest points between "objects" in two raster
maps. An "object" is defined as all the grid cells that have the same
category number, and closest means having the shortest "straight-line"
distance. The cell centers are considered for the distance calculation
(two adjacent grid cells have the distance between their cell centers).

The output is an ascii list, one line per pair of objects, in the
following form:

```sh
from_category:to_category:distance:from_easting:from_northing:to_easting:to_northing
```

**from_category**  
Category number from map1

**to_category**  
Category number from map2

**distance**  
The distance in meters between "from_category" and "to_category"

**from_easting,from_northing**  
The coordinates of the grid cell "from_category" which is closest to "to_category"

**to_easting,to_northing**  
The coordinates of the grid cell "to_category" which is closest to "from_category"

### Flags

**-l** The -l flag outputs the category labels of the matched raster
objects at the beginning of the line, if they exist.

**-o** The -o flag reports zero distance if the input rasters are
overlapping.

## NOTES

The output format lends itself to filtering. For example, to "see" lines
connecting each of the category pairs in two maps, filter the output
using awk and then into *d.graph*:

```sh
r.distance map=map1,map2 | \
  awk -F: '{print "move",$4,$5,"\ndraw",$6,$7}' | d.graph -m
```

To create a vector map of all the "map1" coordinates, filter the output
into awk and then into *v.in.ascii*:

```sh
r.distance map=map1,map2 | \
  awk -F: '{print $4,$5}' | v.in.ascii format=point output=name separator=space
```

## EXAMPLES

### Using JSON output with Pandas to locate closest points

```python
import pandas as pd
import grass.script as gs

result = gs.parse_command(
    "r.distance", map=["zipcodes", "lakes"], flags="l", format="json"
)
df = pd.json_normalize(result)
print(df)
```

Possible output:

```text
        distance  from_cell.category  from_cell.easting  from_cell.northing from_cell.label  to_cell.category  to_cell.easting  to_cell.northing to_cell.label
0   11158.870911               27511             632605              223295            CARY             34300           640585            215495      Dam/Weir
1    1037.304198               27511             631735              222695            CARY             39000           632315            221835     Lake/Pond
2    2277.059507               27511             630015              221605            CARY             43600           630765            219455     Reservoir
..           ...                 ...                ...                 ...             ...               ...              ...               ...           ...
36   4922.600939               27610             642975              219815         RALEIGH             34300           640615            215495      Dam/Weir
37     50.000000               27610             642655              222765         RALEIGH             39000           642705            222765     Lake/Pond
38  11368.069317               27610             642305              220945         RALEIGH             43600           631035            219455     Reservoir
```

## SEE ALSO

*[r.buffer](r.buffer.md), [r.cost](r.cost.md), [r.drain](r.drain.md),
[r.grow](r.grow.md), [r.grow.distance](r.grow.distance.md),
[v.distance](v.distance.md)*

## AUTHOR

Michael Shapiro, U.S. Army Construction Engineering Research Laboratory

## SOURCE CODE

Available at: [r.distance source code](https://github.com/OSGeo/grass/tree/main/raster/r.distance)
([history](https://github.com/OSGeo/grass/commits/main/raster/r.distance))  
Latest change: Thursday Oct 02 14:39:13 2025 in commit [1850489](https://github.com/OSGeo/grass/commit/1850489bf42c72d8dcfe2e4db1738441b4a7dc43)
