---
name: v.distance
description: Finds the nearest element in vector map 'to' for elements in vector map 'from'.
keywords: [ vector, distance, database, attribute table ]
---

# v.distance

Finds the nearest element in vector map 'to' for elements in vector map 'from'.

=== "Command line"

    **v.distance**
    [**-pas**]
    **from**=*name*
    [**from_layer**=*string*]
    [**from_type**=*string* [,*string*,...]]
    **to**=*name*
    [**to_layer**=*string*]
    [**to_type**=*string* [,*string*,...]]
    [**output**=*name*]
    [**dmax**=*float*]
    [**dmin**=*float*]
    **upload**=*string* [,*string*,...]
    [**column**=*name* [,*name*,...]]
    [**to_column**=*name*]
    [**table**=*name*]
    [**separator**=*character*]
    **format**=*name*
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    v.distance from=name to=name output=name upload=cat format=plain
    ```

=== "Python (grass.script)"

    *grass.script.parse_command*("***v.distance***",
        **from**,
        **from_layer**=*"1"*,
        **from_type**=*"point,line,area"*,
        **to**,
        **to_layer**=*"1"*,
        **to_type**=*"point,line,area"*,
        **output**=*None*,
        **dmax**=*-1*,
        **dmin**=*-1*,
        **upload**,
        **column**=*None*,
        **to_column**=*None*,
        **table**=*None*,
        **separator**=*"pipe"*,
        **format**=*"plain"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.parse_command("v.distance", from="name", to="name", output="name", upload="cat", format="json")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.v_distance*(**from**,
        **from_layer**=*"1"*,
        **from_type**=*"point,line,area"*,
        **to**,
        **to_layer**=*"1"*,
        **to_type**=*"point,line,area"*,
        **output**=*None*,
        **dmax**=*-1*,
        **dmin**=*-1*,
        **upload**,
        **column**=*None*,
        **to_column**=*None*,
        **table**=*None*,
        **separator**=*"pipe"*,
        **format**=*"plain"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.v_distance(from="name", to="name", output="name", upload="cat", format="json")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **from**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of existing vector map (from)  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    **from_layer**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name (from)  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **from_type**=*string* [,*string*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Feature type (from)  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *point, line, boundary, centroid, area*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *point,line,area*  
    **to**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of existing vector map (to)  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    **to_layer**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name (to)  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **to_type**=*string* [,*string*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Feature type (to)  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *point, line, boundary, centroid, area*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *point,line,area*  
    **output**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output vector map containing lines connecting nearest elements  
    **dmax**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum distance or -1 for no limit  
    &nbsp;&nbsp;&nbsp;&nbsp;Map units, meters for ll  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-1*  
    **dmin**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum distance or -1 for no limit  
    &nbsp;&nbsp;&nbsp;&nbsp;Map units, meters for ll  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-1*  
    **upload**=*string* [,*string*,...] **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Values describing the relation between two nearest features  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *cat, dist, to_x, to_y, to_along, to_angle, to_attr*  
    &nbsp;&nbsp;&nbsp;&nbsp;**cat**: category of the nearest feature  
    &nbsp;&nbsp;&nbsp;&nbsp;**dist**: minimum distance to nearest feature  
    &nbsp;&nbsp;&nbsp;&nbsp;**to_x**: x coordinate of the nearest point on the 'to' feature  
    &nbsp;&nbsp;&nbsp;&nbsp;**to_y**: y coordinate of the nearest point on the 'to' feature  
    &nbsp;&nbsp;&nbsp;&nbsp;**to_along**: distance to the nearest point on the 'to' feature along that linear feature  
    &nbsp;&nbsp;&nbsp;&nbsp;**to_angle**: angle along the nearest linear feature in the 'to' map, measured CCW from the +x axis, in radians, between -Pi and Pi inclusive  
    &nbsp;&nbsp;&nbsp;&nbsp;**to_attr**: attribute of nearest feature given by to_column option  
    **column**=*name* [,*name*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Column name(s) where values specified by 'upload' option will be uploaded  
    **to_column**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Column name of nearest feature (used with upload=to_attr)  
    **table**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for new attribute table  
    **separator**=*character*  
    &nbsp;&nbsp;&nbsp;&nbsp;Field separator for printing output to stdout  
    &nbsp;&nbsp;&nbsp;&nbsp;Special characters: pipe, comma, space, tab, newline  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *pipe*  
    **format**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Plain text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    **-p**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print output to stdout, don't update attribute table  
    &nbsp;&nbsp;&nbsp;&nbsp;First column is always category of 'from' feature called from_cat  
    **-a**  
    &nbsp;&nbsp;&nbsp;&nbsp;Calculate distances to all features within the threshold  
    &nbsp;&nbsp;&nbsp;&nbsp;Output may be written to stdout using the '-p' flag or uploaded to a new table created by the 'table' option; multiple 'upload' options may be used.  
    **-s**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print output as square matrix (only possible for one single upload variable)  
  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **from** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of existing vector map (from)  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **from_layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name (from)  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **from_type** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Feature type (from)  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *point, line, boundary, centroid, area*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *point,line,area*  
    **to** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of existing vector map (to)  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **to_layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name (to)  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **to_type** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Feature type (to)  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *point, line, boundary, centroid, area*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *point,line,area*  
    **output** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output vector map containing lines connecting nearest elements  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, vector, *name*  
    **dmax** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum distance or -1 for no limit  
    &nbsp;&nbsp;&nbsp;&nbsp;Map units, meters for ll  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-1*  
    **dmin** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum distance or -1 for no limit  
    &nbsp;&nbsp;&nbsp;&nbsp;Map units, meters for ll  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-1*  
    **upload** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Values describing the relation between two nearest features  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *cat, dist, to_x, to_y, to_along, to_angle, to_attr*  
    &nbsp;&nbsp;&nbsp;&nbsp;**cat**: category of the nearest feature  
    &nbsp;&nbsp;&nbsp;&nbsp;**dist**: minimum distance to nearest feature  
    &nbsp;&nbsp;&nbsp;&nbsp;**to_x**: x coordinate of the nearest point on the 'to' feature  
    &nbsp;&nbsp;&nbsp;&nbsp;**to_y**: y coordinate of the nearest point on the 'to' feature  
    &nbsp;&nbsp;&nbsp;&nbsp;**to_along**: distance to the nearest point on the 'to' feature along that linear feature  
    &nbsp;&nbsp;&nbsp;&nbsp;**to_angle**: angle along the nearest linear feature in the 'to' map, measured CCW from the +x axis, in radians, between -Pi and Pi inclusive  
    &nbsp;&nbsp;&nbsp;&nbsp;**to_attr**: attribute of nearest feature given by to_column option  
    **column** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Column name(s) where values specified by 'upload' option will be uploaded  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **to_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Column name of nearest feature (used with upload=to_attr)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **table** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for new attribute table  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: dbtable, *name*  
    **separator** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Field separator for printing output to stdout  
    &nbsp;&nbsp;&nbsp;&nbsp;Special characters: pipe, comma, space, tab, newline  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, separator, *character*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *pipe*  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Plain text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *p*, *a*, *s*  
    &nbsp;&nbsp;&nbsp;&nbsp;**p**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print output to stdout, don't update attribute table  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;First column is always category of 'from' feature called from_cat  
    &nbsp;&nbsp;&nbsp;&nbsp;**a**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Calculate distances to all features within the threshold  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Output may be written to stdout using the '-p' flag or uploaded to a new table created by the 'table' option; multiple 'upload' options may be used.  
    &nbsp;&nbsp;&nbsp;&nbsp;**s**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print output as square matrix (only possible for one single upload variable)  
  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **from** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of existing vector map (from)  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **from_layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name (from)  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **from_type** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Feature type (from)  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *point, line, boundary, centroid, area*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *point,line,area*  
    **to** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of existing vector map (to)  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **to_layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name (to)  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **to_type** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Feature type (to)  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *point, line, boundary, centroid, area*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *point,line,area*  
    **output** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output vector map containing lines connecting nearest elements  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, vector, *name*  
    **dmax** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum distance or -1 for no limit  
    &nbsp;&nbsp;&nbsp;&nbsp;Map units, meters for ll  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-1*  
    **dmin** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum distance or -1 for no limit  
    &nbsp;&nbsp;&nbsp;&nbsp;Map units, meters for ll  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-1*  
    **upload** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Values describing the relation between two nearest features  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *cat, dist, to_x, to_y, to_along, to_angle, to_attr*  
    &nbsp;&nbsp;&nbsp;&nbsp;**cat**: category of the nearest feature  
    &nbsp;&nbsp;&nbsp;&nbsp;**dist**: minimum distance to nearest feature  
    &nbsp;&nbsp;&nbsp;&nbsp;**to_x**: x coordinate of the nearest point on the 'to' feature  
    &nbsp;&nbsp;&nbsp;&nbsp;**to_y**: y coordinate of the nearest point on the 'to' feature  
    &nbsp;&nbsp;&nbsp;&nbsp;**to_along**: distance to the nearest point on the 'to' feature along that linear feature  
    &nbsp;&nbsp;&nbsp;&nbsp;**to_angle**: angle along the nearest linear feature in the 'to' map, measured CCW from the +x axis, in radians, between -Pi and Pi inclusive  
    &nbsp;&nbsp;&nbsp;&nbsp;**to_attr**: attribute of nearest feature given by to_column option  
    **column** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Column name(s) where values specified by 'upload' option will be uploaded  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **to_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Column name of nearest feature (used with upload=to_attr)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **table** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for new attribute table  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: dbtable, *name*  
    **separator** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Field separator for printing output to stdout  
    &nbsp;&nbsp;&nbsp;&nbsp;Special characters: pipe, comma, space, tab, newline  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, separator, *character*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *pipe*  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Plain text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *p*, *a*, *s*  
    &nbsp;&nbsp;&nbsp;&nbsp;**p**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print output to stdout, don't update attribute table  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;First column is always category of 'from' feature called from_cat  
    &nbsp;&nbsp;&nbsp;&nbsp;**a**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Calculate distances to all features within the threshold  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Output may be written to stdout using the '-p' flag or uploaded to a new table created by the 'table' option; multiple 'upload' options may be used.  
    &nbsp;&nbsp;&nbsp;&nbsp;**s**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print output as square matrix (only possible for one single upload variable)  
  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*v.distance* finds the nearest element in vector map (**to**) for
elements in vector map (**from**). Various information about the
vectors' relationships (distance, category, etc.) may be uploaded to the
attribute table attached to the first vector map, or printed to
'stdout'. A new vector map may be created where lines connecting nearest
points on features are written. **dmin** and/or **dmax** can be used to
limit the search radius (in lat-long projects to be given in meters
since they are calculated as geodesic distances on a sphere).

For lines to lines, say line A to line B, *v.distance* calculates the
shortest distance of each vertex in A with each segment (not vertex) in
B. The module then calculates the shortest distance of each vertex in B
to each segment in A. The overall shortest distance of A points to B
segments and B points to A segments is used. Additionally, *v.distance*
checks for intersections. In case of intersections, the first
intersection found is used and the distance set to zero.

For lines to areas, the distance is set to zero if a line is (partially)
inside an area. The first point of the line that is inside the area is
used as common point. The distance is also set to zero if the line
intersects with the outer ring or any of the inner rings (isles), in
which case the fist intersection is used as common point.

For areas to areas, the module checks first for overlap or if one area
is (partially) inside the other area. This is computationally quite
intensive. If the outer rings of the two areas do not overlap, the
distance is calculated as above for lines to lines, treating the outer
rings as two lines. Again, the first point encountered falling into an
area is used as common point, or the first intersection point.

For anything else than points to lines, there can be several common
locations with zero distance, and the common location would then be the
result of an overlay consisting of several points, lines, or areas.
*v.distance* selects in these cases a single point, and does not create
an overlay like *[v.overlay](v.overlay.md)*. In this implementation, any
shared point is as good as any other. Calculating an intersection is
costlier than to check if a vertex is inside a polygon. For example, if
a vertex of the boundary of the 'to' area is inside the 'from' area, it
is a common location. For speed reasons, the distance is then set to
zero and no further tests are done.

## NOTES

If a nearest feature does not have a category, the attribute column is
updated to NULL.

The upload **column**(s) must already exist. Create one with
*[v.db.addcolumn](v.db.addcolumn.md)*.

In lat-long projects *v.distance* gives distances (*dist*, *from_along*,
and *to_along*) not in degrees but in meters calculated as geodesic
distances on a sphere.

If one or both of the input vector maps are 3D, the user is notified
accordingly.

The *-p* flag prints the results to standard output. By default the
output is in form of a linear matrix. If only only variable is uploaded
and a square matrix is desired, the user can set the *-s* flag.

## EXAMPLES

### Find nearest lines

Find *nearest lines* in vector map "ln" for points from vector map "pnt"
within the given threshold and write related line categories to column
"linecat" in an attribute table attached to vector map "pnt":

```sh
v.distance from=pnt to=ln upload=cat column=linecat
```

### Find nearest area

For each point from vector map "pnt", find the *nearest area* from map
"ar" within the given threshold and write the related area categories to
column "areacat" in an attribute table attached to vector map "pnt" (in
the case that a point falls into an area, the distance is zero):

```sh
v.distance from=pnt to=ar upload=cat column=areacat
```

### Create a new vector map

Create a new vector map which contains *lines connecting nearest
features* of maps "pnt" and map "ln". The resulting vector map can be
used for example to connect points to a network as needed for network
analysis:

```sh
v.distance from=pnt to=ln out=connections upload=dist column=dist
```

### Create a new vector map with from and to categories in the attribute table

Create a new vector map that contains *lines connecting nearest
features* of maps "pnt" and map "ln", and a new attribute table that
contains distances, from and to categories from the input maps:

```sh
v.distance from=pnt to=ln out=connections upload=cat,dist column=to_cat,dist table=connections
```

### Query information

Query information from selected point(s). *v.distance* takes points from
a vector map as input instead of stdin. A new vector map with query
points has to be created before the map can be analysed.

Create query map (if not present):

```sh
echo "123456|654321|1" | v.in.ascii output=pnt
```

Find nearest features:

```sh
v.distance -p from=pnt to=map_to_query upload=cat
```

### Point-in-polygon

The option **dmax=0** is here important because otherwise for points not
falling into any area, the category of the nearest area is recorded.  
For each point from vector map "pnt", find the *area* from vector map
"ar" in which the individual point falls, and write the related area
categories to column "areacat" into the attribute table attached to
vector map "pnt":

```sh
v.distance from=pnt to=ar dmax=0 upload=cat column=areacat
```

### Univariate statistics on results

Create a vector map containing connecting lines and investigate mean
distance to targets. An alternative solution is to use the
`v.distance upload=dist` option to upload distances into the *bugs*
vector directly, then run v.univar on that. Also note you can upload two
columns at a time, e.g.
`v.distance upload=cat,dist column=nearest_id,dist_to_nr`.

```sh
# create working copy
g.copy vect=bugsites,bugs

# add new attribute column to hold nearest archsite category number
v.db.addcolumn map=bugs column="nrst_arch INTEGER"

v.distance from=bugs to=archsites to_type=point upload=to_attr \
  to_column=cat column=nrst_arch out=vdistance_vectors_raw

# we need to give the lines category numbers, create a table, and create
#  a column in that table to hold the distance data.
v.category vdistance_vectors_raw out=vdistance_vectors type=line op=add
g.remove -f type=vector name=vdistance_vectors_raw

v.db.addtable map=vdistance_vectors column="length DOUBLE"
v.to.db map=vdistance_vectors option=length column=length

# calculate statistics
v.univar vdistance_vectors column=length
```

### Print distance between points

Example for a Latitude-longitude project (EPSG 4326):

```sh
# points along the equator
echo "0|-61|1" | v.in.ascii output=pnt1 input=-
echo "0|-58|1" | v.in.ascii output=pnt2 input=-

# here, distances are in degree units
v.distance -p --q from=pnt1 to=pnt2 upload=dist
from_cat|distance
1|3
```

### Print distance matrix

North Carolina sample data

As linear matrix:

```sh
v.distance -pa from=hospitals to=hospitals upload=dist,to_attr to_column=NAME separator=tab
from_cat    to_cat  dist    to_attr
1   1   0   Cherry Hospital
1   2   7489.1043632983983  Wayne Memorial Hospital
1   3   339112.17046729225  Watauga Medical Center
1   4   70900.392145909267  Central Prison Hospital
1   5   70406.227393921712  Dorothea Dix Hospital
```

As square matrix (only possible with single upload option):

```sh
v.distance -pas from=hospitals to=hospitals upload=dist separator=tab
from_cat to_cat       dist
              1          2          3          4          5 ...
1             0    7489.10  339112.17   70900.39   70406.23 ...
2       7489.10          0  345749.12   76025.46   75538.87 ...
3     339112.17  345749.12          0  274153.19  274558.98 ...
4      70900.39   76025.46  274153.19          0     501.11 ...
5      70406.23   75538.87  274558.98     501.11          0 ...
...
```

### Print in JSON

```sh
v.distance -p from=busroute_a to=busstopsall upload=dist,to_attr to_column=routes format=json
```

Possible output:

```json
[
    {
        "from_cat": 1,
        "to_cat": 112,
        "dist": 0.1428123184481199,
        "to_attr": "8,A"
    },
    {
        "from_cat": 2,
        "to_cat": 44,
        "dist": 0.10232660032693719,
        "to_attr": "9,A"
    }
]
```

### Print output as Pandas DataFrame

```python
import grass.script as gs
import pandas as pd

data = gs.parse_command(
    "v.distance",
    flags="p",
    from_="busroute_a",
    to="busstopsall",
    upload="dist,to_attr",
    to_column="routes",
    format="json",
)
df = pd.DataFrame(data)

print(df)
```

Possible output:

```text
   from_cat  to_cat      dist to_attr
0         1     112  0.142812     8,A
1         2      44  0.102327     9,A
```

## SEE ALSO

*[r.distance](r.distance.md), [v.db.addcolumn](v.db.addcolumn.md),
[v.what.vect](v.what.vect.md)*

## AUTHORS

Janne Soimasuo 1994, University of Joensuu, Faculty of Forestry,
Finland  
Cmd line coordinates support: Markus Neteler, ITC-irst, Trento, Italy  
Updated for 5.1: Radim Blazek, ITC-irst, Trento, Italy  
Matrix-like output by Martin Landa, FBK-irst, Trento, Italy  
Improved processing speed: Markus Metz  
Distance from any feature to any feature: Markus Metz  
New table without the -p flag: Huidae Cho Make linear matrix the default
for all outputs: Moritz Lennert

## SOURCE CODE

Available at: [v.distance source code](https://github.com/OSGeo/grass/tree/main/vector/v.distance)
([history](https://github.com/OSGeo/grass/commits/main/vector/v.distance))  
Latest change: Thursday Oct 02 14:39:13 2025 in commit [1850489](https://github.com/OSGeo/grass/commit/1850489bf42c72d8dcfe2e4db1738441b4a7dc43)
